/*
 * syslogd.h - syslogd implementation for windows, common definitions
 *
 * Created by Alexander Yaworsky
 *
 * THIS SOFTWARE IS NOT COPYRIGHTED
 *
 * This source code is offered for use in the public domain. You may
 * use, modify or distribute it freely.
 *
 * This code is distributed in the hope that it will be useful but
 * WITHOUT ANY WARRANTY. ALL WARRANTIES, EXPRESS OR IMPLIED ARE HEREBY
 * DISCLAIMED. This includes but is not limited to warranties of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

extern int verbosity_level;
extern void display_message( FILE* fd, char* file, int line, const char* func, char* fmt, ... );

#ifdef HAVE_DEBUG
#  define DO_TRACE( verbosity, fmt... ) \
       do { \
           if( verbosity <= verbosity_level ) \
               display_message( stderr, __FILE__, __LINE__, __FUNCTION__, fmt ); \
       } while(0)
#  define TRACE_2( fmt... ) DO_TRACE( 2, fmt )
#  define TRACE( fmt... ) DO_TRACE( 1, fmt )
#else
#  define TRACE_2( fmt... )
#  define TRACE( fmt... )
#endif
#define TRACE_ENTER TRACE_2
#define TRACE_LEAVE TRACE_2
#define ERR( fmt... ) display_message( stderr, __FILE__, __LINE__, __FUNCTION__, fmt )

/* refcounted string */
struct string
{
    LONG refcount;
    GString *gstr;
};

extern struct string* string_new( const gchar* init );
extern struct string* string_new_len( const gchar* init, gssize len );
extern struct string* string_addref( struct string* s );
extern void string_release( struct string* s );
extern struct string* string_vprintf( gchar* fmt, va_list args );
extern struct string* string_printf( gchar* fmt, ... );
extern gsize string_concat( gchar** result, struct string* s, ... );
extern gboolean string_equal( struct string* s1, struct string* s2 );

/* misc global stuff */
extern HANDLE service_stop_event;

extern struct string *local_hostname;
extern struct string *self_program_name;
extern struct string *space;
extern struct string *line_feed;

extern char *str_month[];

extern void syslogd_main();

extern void log_internal( int pri, char* fmt, ... );

/* options and their default values */
extern gboolean use_dns;
extern gchar *source_encoding;
extern gchar *destination_encoding;
extern int mark_interval;
extern gchar *mark_message;
extern int hold;
extern gchar *logdir;

/* listener */
struct raw_message
{
    gchar *msg;
    struct sockaddr_in sender_addr;
    struct source *source;
};

extern struct fifo *udp_message_queue;
extern HANDLE udp_queue_event;
extern CRITICAL_SECTION udp_queue_cs;

extern gboolean init_udp_listener();
extern void shutdown_udp_listener();
extern void fini_udp_listener();

/* message */
struct message
{
    LONG refcount;
    struct source *source;
    struct string *sender;
    int facility;
    int priority;
    struct string *timestamp;
    struct string *hostname;
    struct string *program;
    struct string *message;
};

extern struct message* create_message( struct source* source,
                                       struct string* sender,
                                       int facility, int priority,
                                       LPSYSTEMTIME timestamp,
                                       struct string* hostname,
                                       struct string* program,
                                       struct string* message );
extern struct message* duplicate_message( struct message* msg );
extern void reference_message( struct message* msg );
extern void release_message( struct message* msg );

/* sources, destinations, filters and logpaths */
enum source_type
{
    ST_UNDEFINED,
    ST_INTERNAL,
    ST_UDP
};

struct source
{
    gchar *name;
    enum source_type type;
    struct sockaddr_in udp;
};

extern unsigned number_of_sources( enum source_type type );

enum destination_type
{
    DT_UNDEFINED,
    DT_FILE,
    DT_RELAY
};

enum rotation_period
{
    RP_UNDEFINED = 0,
    RP_INVALID,
    RP_DAILY,
    RP_WEEKLY,
    RP_MONTHLY
};

struct destination_file
{
    gchar *name_pattern;
    enum rotation_period rotate;
    int size;
    int backlogs;
    gboolean ifempty;
    gchar *olddir;
    gchar *compresscmd;
    gchar *compressoptions;
};

struct destination_relay
{
    gchar *collector;
    gboolean omit_hostname;
};

struct destination;

typedef void (*dest_put)( struct destination* destination, struct message* message );
typedef void (*dest_finalize)( struct destination* destination );

struct destination
{
    gchar *name;
    enum destination_type type;
    union
    {
      struct destination_file file;
      struct destination_relay relay;
    } u;
    void *extra;
    /* methods */
    dest_put put;
    dest_finalize fini;
};

struct filter
{
    gchar *name;
    gboolean facilities[ LOG_NFACILITIES ];
    gboolean priorities[ 8 ];
};

struct logpath
{
    struct source *source;
    struct filter *filter;
    struct destination *destination;
};

extern GList *sources;
extern GList *destinations;
extern GList *filters;
extern GList *logpaths;

extern gboolean read_configuration();

extern gboolean init_destination_file( struct destination* destination );
extern gboolean init_destination_relay( struct destination* destination );

/* queue */
struct fifo;

extern struct fifo* fifo_create();
extern void fifo_destroy( struct fifo* queue );
extern gboolean fifo_push( struct fifo* queue, void* data );
extern void* fifo_pop( struct fifo* queue );

/* logrotate */
extern void rotate_logfile( const gchar* pathname, struct destination* destination );

/* purger */
struct purger_dir
{
    gchar *directory;
    int keep_days;
};

extern GList *purger_dirs;

extern gboolean init_purger();
extern void fini_purger();
extern void purge_log_dirs();

/* pathnames */
extern gchar* make_absolute_log_pathname( char* path_appendix );
extern void create_directories( gchar* pathname );
extern gchar* normalize_pathname( const gchar* pathname );

/* workaround for syslog.h: included with SYSLOG_NAMES in names.c */
typedef struct _code {
	char	*c_name;
	int	c_val;
} CODE;

extern CODE prioritynames[];
extern CODE facilitynames[];

extern char* get_priority_name( int pri );
extern char* get_facility_name( int pri );
