#define _GNU_SOURCE
#include "config.h"

#include <getopt.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include "optinfo.h"

#ifndef PLCONV_BUGREPORT
#  define PLCONV_BUGREPORT "Bartosz Lis <bartoszl@icsp.lodz.pl>"
#endif /*PLCONV_BUGREPORT */
#ifndef PLCONV_STRING
#  define PLCONV_STRING PACKAGE " " VERSION
#endif /*PLCONV_STRING*/

#define NON_EMPTY(str) ((str) && *(str))

#define EXE_PREFIX "pl-"
#define EXE_INFIX  "2"

#define BEGIN_CP_DEFS(defs) \
  const unsigned char code_pages[][19] =\
  { \
    defs \
  };
#define DEFINE_CP(name,c1,c2,c3,c4,c5,c6,c7,c8,c9,C1,C2,C3,C4,C5,C6,C7,C8,C9,defs) \
    { c1,c2,c3,c4,c5,c6,c7,c8,c9,C1,C2,C3,C4,C5,C6,C7,C8,C9,0 },\
    defs
#define END_CP_DEFS \
    { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }

#include "chardefs.h"

#undef BEGIN_CP_DEFS
#undef DEFINE_CP
#undef END_CP_DEFS

#define BEGIN_CP_DEFS(defs) \
  enum enum_code_pages \
  { \
    defs \
  };
#define DEFINE_CP(name,c1,c2,c3,c4,c5,c6,c7,c8,c9,C1,C2,C3,C4,C5,C6,C7,C8,C9,defs) \
    id_cp_ ## name,\
    defs
#define END_CP_DEFS \
    num_cp

#include "chardefs.h"

int cp_index=-1;

struct optinfo_t opt_main[] =
{
  { 'h', "help",    "wyswietla liste opcji", 0, 0, 'h' },
  { 'v', "version", "wyswietla pen nazwe programu wesje i autora", 0, 0, 'v' },
  { 'i', "input",   "plik wejsciowy (jesli nie uzyto, dane brane sa ze stdin)", "file", 0, 'i' },
  { 'o', "output",  "plik wyjsciowy (jesli nie uzyto, dane ida do stdout)", "file", 0, 'o' }
};

#define OPT_MAIN_SIZE (sizeof(opt_main)/sizeof(struct optinfo_t))

struct optinfo_t opt_trans[] =
{
  { 'I', "iso8859-2",    "norma ISO 8859, strona kodowa nr 2",
    0, &cp_index, id_cp_iso
  },
  { 'W', "windows",      0,
    0, &cp_index, id_cp_win
  },
  { 0,   "cp1250",       "Microsoft Code Page 1250 (uzywana w MS Windows)",
    0, &cp_index, id_cp_win
  },
  { 'D', "dos",          0,
    0, &cp_index, id_cp_dos
  },
  { 0,   "cp852",        "Microsoft Code Page 852 (uzywana w MS DOS)",
    0, &cp_index, id_cp_dos
  },
  { 'M', "mazovia",      "najpopularniejszy sposb kodowania w MS DOSie",
    0, &cp_index, id_cp_mazovia
  },
  { 0,   "pnb",          "Polska Norma Branzowa PNB 74/3101-01",
    0, &cp_index, id_cp_old_pnb
  },
  { 0,   "dhn",          "Dom Handlowy Nauki",
    0, &cp_index, id_cp_dhn
  },
  { 0,   "cyfromat",     "Cyfromat",
    0, &cp_index, id_cp_cyfromat
  },
  { 0,   "csk",          "Computer Studio Kajkowski",
    0, &cp_index, id_cp_csk
  },
  { 0,   "microvex",     "Microvex",
    0, &cp_index, id_cp_microvex
  },
  { 0,   "ventura-old",  "stary Ventura Publisher",
    0, &cp_index, id_cp_old_ventura
  },
  { 'E', "elwro-800-jr", "Elwro 800 JR - CP/J",
    0, &cp_index, id_cp_cpj
  },
  { 'P', "polskawe",     "lacinskie odpowiedniki polskich znakow",
    0, &cp_index, id_cp_polskawe
  },
};

#define OPT_TRANS_SIZE (sizeof(opt_trans)/sizeof(struct optinfo_t))

void
error_msg (FILE *f, char *progname, const char *msg, int usage, int code)
{
  if (msg) fprintf(f, "%s: %s\n",progname,msg);
  else fprintf(f, PLCONV_STRING "\n"
                  "Przekodowywanie polskich znakw/Polish diacritic character conversion\n"
                  "Copyright (C) 1991 - 2002 " PLCONV_BUGREPORT "\n"
                  "This program is distributed under the GNU General Public License\n",
                  "For more information see file COPYING in its sources\n");
  if (usage)
  {
    fprintf(f, "\nUzycie/Usage: %s [<opcje>] --<kod_we> --<kod_wy>\n"
               "Glowne opcje:\n", progname);
    FPRINTF_OPTIONS(f, opt_main);
    fprintf(f, "okreslenie stron kodowych wejsciowa i wyjciowa (dopuszcza sie skracanie):\n");
    FPRINTF_OPTIONS(f, opt_trans);
    fprintf(f, "\nMozna dowiazac plik wykonywalny do nazw postaci " EXE_PREFIX "<kod_we>" EXE_INFIX "<kod_wy>\n"
               "np. " EXE_PREFIX "win " EXE_INFIX "iso, co upraszcza wywolywanie.\n");
  }
  fflush(0);
  exit(code);
}

struct prog_options
{
  char *f_in;
  char *f_out;
  int   cp_in;
  int   cp_out;
  int   inout;
};

void
set_cp(int id_cp,char *progname,struct prog_options *opt)
{
  switch (opt->inout)
  {
  case 0: opt->cp_in=id_cp; break;
  case 1: opt->cp_out=id_cp; break;
  default: error_msg(stderr,progname,"specify only input and output codepages - podaj tylko wejsciowa i wyjsciowa strone kodowa",1,3);
  }
  ++opt->inout;
}

void
check_arg (int argc, char **argv, int check_all,
           struct prog_options *opt)
{
  struct  option opt_arr[OPT_MAIN_SIZE+OPT_TRANS_SIZE+1];
  size_t  len;
  char   *optstring, *str;
  int     id;
  opt->inout=0;
  len=init_long_options(opt_arr,opt_trans,OPT_TRANS_SIZE);
  if (check_all)
    len+=init_long_options(opt_arr+OPT_TRANS_SIZE,opt_main,OPT_MAIN_SIZE);
  if (!(optstring=(char *)malloc(len+1)))
    error_msg(stderr,argv[0],"short on memory - brak pamieci",0,2);
  str=init_short_options(optstring,opt_trans,OPT_TRANS_SIZE);
  if (check_all) str=init_short_options(str,opt_main,OPT_MAIN_SIZE);
  *str=0;
  optind=1;
  while ((id=getopt_long(argc,argv,optstring,opt_arr,0))!=-1) switch(id)
  {
  case 0:
    if (cp_index==-1)
      error_msg(stderr,argv[0],"unknown option - nie rozpoznana opcja",1,3);
    set_cp(cp_index,argv[0],opt);
    cp_index=-1;
    break;
  case 'I':
    set_cp(id_cp_iso,argv[0],opt);
    break;
  case 'W':
    set_cp(id_cp_win,argv[0],opt);
    break;
  case 'D':
    set_cp(id_cp_dos,argv[0],opt);
    break;
  case 'M':
    set_cp(id_cp_mazovia,argv[0],opt);
    break;
  case 'E':
    set_cp(id_cp_cpj,argv[0],opt);
    break;
  case 'P':
    set_cp(id_cp_polskawe,argv[0],opt);
    break;
  case 'i':
    if (opt->f_in)
      error_msg(stderr,argv[0],"input file specified more than once - plik wejsciowy podany wiecej niz raz",1,3);
    opt->f_in=optarg;
    break;
  case 'o':
    if (opt->f_out)
      error_msg(stderr,argv[0],"output file specified more than once - plik wyjsciowy podany wiecej niz raz",1,3);
    opt->f_out=optarg;
    break;
  case 'h':
    error_msg(stdout,argv[0],0,1,0);
  case 'v':
    error_msg(stdout,argv[0],0,0,0);
  case ':':
    error_msg(stderr,argv[0],"missing parameter for an option - jednej z opcji brakuje parametru",1,3);
  default:
    error_msg(stderr,argv[0],"unknown option - nie rozpoznana opcja",1,3);
  }
  if (argv[optind])
    error_msg(stderr,argv[0],"unknown argument - nie rozpoznany argument",1,3);
}

void
check_prog (char *progname, struct prog_options *opt)
{
  char   *argv[4], *part1, *part2, *arg1, *arg2;
  size_t  len1, len2;
  if (part1=strrchr(progname,'/')) ++part1;
  else part1=progname;
  if (strncmp (part1, EXE_PREFIX, sizeof(EXE_PREFIX)-1)) return;
  part1+=sizeof(EXE_PREFIX)-1;
  if (!(part2=strstr (part1,EXE_INFIX))) return;
  if (!(len1=part2-part1)) return;
  part2+=sizeof(EXE_INFIX)-1;
  if (!(len2=strlen (part2))) return;
  if (!(arg1=(char *)malloc(len1+3))
      || !(arg2=(char *)malloc(len2+3)))
    error_msg(stderr,progname,"short on memory - brak pamieci",0,2);
  arg1[0]=arg1[1]=arg2[0]=arg2[1]='-';
  strncpy (arg1+2,part1,len1);
  strncpy (arg2+2,part2,len2);
  arg1[len1+2]=arg2[len2+2]=0;
  argv[0]=progname;
  argv[1]=arg1;
  argv[2]=arg2;
  argv[3]=0;
  check_arg (3,argv,0,opt);
  free(arg1);
  free(arg2);
}

int
translate(FILE *f_in, FILE *f_out, const char *cp_in, const char *cp_out)
{
  char        c;
  const char *p;
  for (;;)
  {
    if ((c=fgetc(f_in))==EOF) return errno;
    if (p=strchr(cp_in,c)) c=cp_out[p-cp_in];
    if (fputc(c,f_out)==EOF) return errno;
  }
}

int
main (int argc, char **argv)
{
  FILE                *fin=0, *fout=0;
  struct prog_options  opt = { 0, 0, -1, -1 };
  check_prog(argv[0],&opt);
  check_arg(argc,argv,1,&opt);
  if (opt.cp_in==id_cp_polskawe)
    error_msg(stderr,argv[0],"cannot convert from polskawe - nie moge konwertowac polskawego tekstu",1,3);
  if (opt.cp_out==-1)
    error_msg(stderr,argv[0],"specify input and output codepage - podaj wejsciowa i wyjsciowa strone kodowa",1,3);
  if (opt.f_in && !(fin=fopen(opt.f_in,"r")))
    error_msg(stderr,argv[0],"cannot open input file - nie moge otworzyc pliku wejsciowego",0,1);
  if (opt.f_out && !(fout=fopen(opt.f_out,"w")))
    error_msg(stderr,argv[0],"cannot open output file - nie moge otworzyc pliku wyjsciowego",0,1);
  if (translate((fin ? fin : stdin),(fout ? fout : stdout),
                code_pages[opt.cp_in], code_pages[opt.cp_out]))
    error_msg(stderr,argv[0],"IO error - blad we/wy",0,1);
  if (fin) fclose(fin);
  if (fout) fclose(fout);
  return 0;
}
