/*
 * Copyright (C) 2002 Bartosz Lis <bartoszl@ics.p.lodz.pl>
 * This module is used to initialize random seed
 */

#define _GNU_SOURCE
#include "config.h"

#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "initseed.h"

#define INIT_BLOCK_SIZE sizeof(unsigned int)

static void
init_with_block(unsigned int *seed, const void *block, size_t size)
{
   unsigned int  buff[2];
   char         *ptr;
   size_t        i;
   for (i=0, ptr=(char *)buff; i<INIT_BLOCK_SIZE; i+=size, ptr+=size)
     memcpy(ptr,block,size);
   *seed+=buff[0];
}

void
init_with_data(unsigned int *seed, const void *data, size_t size)
{
  const char *ptr=(const char *)data;
  size_t      s=INIT_BLOCK_SIZE;
  while (size>0)
    {
       if (s>size) s=size;
       init_with_block(seed,ptr,s);
       size-=s;
       ptr+=s;
    }
}

void
init_with_string(unsigned int *seed, const char *string)
{
   const char *ptr;
   size_t      s;
   if (string) while (*string)
     {
        for (s=0, ptr=string; (s<INIT_BLOCK_SIZE) && *ptr; ++s, ++ptr);
        init_with_block(seed,ptr,s);
        string=ptr;
     }
}

volatile int chld_fin;
int          chld_pid;

static void
sigchld_handler(int sig)
{
   if (!chld_fin && (waitpid(chld_pid,0,WNOHANG)==chld_pid)) chld_fin=1;
}

static void
child (int fd, char *command)
{
   char *argv[] =
     {
        SHELL,
#ifdef SHELL_COMMAND_OPTION
        SHELL_COMMAND_OPTION,
#endif /* SHELL_COMMAND_OPTION */
        command,
        0,
     };
   dup2(fd,1);
   dup2(fd,2);
   close(fd);
   execv(SHELL, argv);
   exit(1);
}

static int
parent (int fd, unsigned int *seed)
{
   int     ret;
   char    buff[INIT_BLOCK_SIZE];
   void    (*old_handler)(int);
   FILE   *f;
   size_t  s;
   chld_fin=0;
   old_handler=signal(SIGCHLD,sigchld_handler);
   if (!(f=fdopen(fd,"r"))) ret=errno;
   else
     {
        while (s=fread(buff,1,INIT_BLOCK_SIZE,f))
          init_with_block(seed,buff,s);
        ret=(feof(f) ? 0 : ferror(f));
        fclose(f);
     }
   if (!chld_fin)
     {
        kill(chld_pid,SIGKILL);
        while ((waitpid(chld_pid,0,0)!=-1) || (errno!=ECHILD));
     }
   signal(SIGCHLD,old_handler);
   return ret;
}

int
init_with_commnd(unsigned int *seed, char *command)
{
   int fd[2];
   if (pipe(fd)<0) return errno;
   switch (chld_pid=fork())
     {
      case -1: return errno;
      case 0:
        close(fd[0]);
        child(fd[1],command);
        return errno;
      default:
        close(fd[1]);
        return parent(fd[0],seed);
     }
}
