/*
 * Copyright (C) 2002 Bartosz Lis <bartoszl@ics.p.lodz.pl>
 * This is the main module
 */

#define _GNU_SOURCE
#include "config.h"

#include <stdio.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <stdlib.h>
#include <termio.h>
#include <security/_pam_types.h>
#include <ctype.h>
#include <pwd.h>
#include <sys/types.h>
#include <getopt.h>

#include "initseed.h"
#include "setpass.h"

#define MAX_USER     255
#define MAX_PASS     255
#define RND_SIZE     256

#ifndef RNDPASS_BUGREPORT
#  define RNDPASS_BUGREPORT "Bartosz Lis <bartoszl@icsp.lodz.pl>"
#endif /*RNDPASS_BUGREPORT */
#ifndef RNDPASS_STRING
#  define RNDPASS_STRING PACKAGE " " VERSION
#endif /*RNDPASS_STRING*/

char *progname;

void
error_msg (FILE *f, const char *msg, int usage, int code)
{
   if (msg) fprintf(f, "%s: %s\n",progname,msg);
   else fprintf(f, RNDPASS_STRING "\n"
                   "Generating random passwords\n"
                   "Copyright (C) 1997 - 2002 " RNDPASS_BUGREPORT "\n"
                   "This program is distributed under the GNU General Public License\n",
                   "For more information see the file COPYING in its sources\n");
   if (usage)
     {
        fprintf(f, "\nUsage: %s [-h | -v | { [<options>] [[-u] <user>]... }\n"
                   "where: -h, --help                 -- help screen\n"
                   "       -v, --version              -- version\n"
                   "       -u, --user <user>          -- change <user> password\n"
                   "                                     this option is valid for root only\n"
                   "options:\n"
                   "-L, --length <length>     -- length of the generated password\n"
                   "-U, --upper <num_upper>   -- number of upper letters in resulting password\n"
                   "-V, --vawels <string>     -- use <string> as vawels\n"
                   "-C, --consonants <string> -- use <string> as consonants\n"
                   "-q, --quiet               -- print only password\n"
                   "-s, --show                -- don't update, print only\n",
                progname);
     }
   fflush(0);
   exit(code);
}

struct option options[] =
{
  { "help",       no_argument,       0, 'h' },
  { "version",    no_argument,       0, 'v' },
  { "user",       required_argument, 0, 'u' },
  { "length",     required_argument, 0, 'P' },
  { "upper",      required_argument, 0, 'U' },
  { "vawels",     required_argument, 0, 'V' },
  { "consonants", required_argument, 0, 'C' },
  { "show",       no_argument,       0, 's' },
  { "quiet",      no_argument,       0, 'q' },
  { 0, 0, 0, 0 }
};

struct user {
  struct user *next;
  char         name[1];
};

struct optval {
  size_t       len;
  size_t       upper;
  const char  *vow;
  const char  *cns;
  short        quiet;
  short        show;
  struct user *first;
  struct user *last;
};

void 
opt_init(struct optval *opt)
{
  opt->len=RNDPASS_LENGTH;
  opt->upper=RNDPASS_UPPER;
  opt->vow=RNDPASS_VOWELS;
  opt->cns=RNDPASS_CONSONANTS;
  opt->quiet=0;
  opt->show=0;
  opt->first=0;
  opt->last=0;
}

void
opt_add_user(struct optval *opt, const char *user)
{
  size_t       len;
  struct user *next;
  len=strlen(user);
  if (!(next=(struct user *)malloc(sizeof(struct user)+len)))
    error_msg(stderr,"memory too short",0,2);
  next->next=0;
  strcpy(next->name,user);
  if (opt->last) opt->last->next=next; else opt->first=next;
  opt->last=next;
}

char *
opt_get_user(struct optval *opt)
{
  return opt->first ? opt->first->name : 0;
}

void
opt_next_user(struct optval *opt)
{
  struct user *head;
  if (head=opt->first)
  {
    if (!(opt->first=head->next)) opt->last=0;
    free(head);
  }
}

void
opt_parse(struct optval *opt, int argc, char **argv)
{
  char     c;
  unsigned x;
  progname=argv[0];
  opt_init(opt);
  while ((c=getopt_long(argc,argv,"hvu:L:U:V:C:qs",options,0))!=-1) switch (c)
  {
  case 'u':
    opt_add_user(opt,optarg);
    break;
  case 'L':
    if (sscanf(optarg,"%u",&x)!=1)
      error_msg(stderr,"cannot parse number of letters",1,3);
    opt->len=x;
    break;
  case 'U':
    if (sscanf(optarg,"%u",&x)!=1)
      error_msg(stderr,"cannot parse number of upper letters",1,3);
    opt->upper=x;
    break;
  case 'V':
    opt->vow=strdup(optarg);
    break;
  case 'C':
    opt->cns=strdup(optarg);
    break;
  case 'q':
    opt->quiet=1;
    break;
  case 's':
    opt->show=1;
    break;
  case 'h':
    error_msg(stdout,0,1,0);
  case 'v':
    error_msg(stdout,0,0,0);
  case ':':
    error_msg(stderr,"missing parameter for an option",1,3);
  default:
    error_msg(stderr,"unknown option",1,3);
  }
  while (optind<argc) opt_add_user(opt,argv[optind++]);
}

void
opt_check(struct optval *opt)
{
  struct passwd *pwd;
  struct user   *user;
  uid_t          uid;
  if (!opt->len)
    error_msg(stderr,
              "password should consist with at least one character",1,3);
  if (MAX_PASS<opt->len)
    error_msg(stderr,"cannot generate passwords of such a length",1,3);
  if (opt->len<opt->upper)
    error_msg(stderr,"more upper letters required than password length",1,3);
  if (!opt->vow[0])
    error_msg(stderr,"specify at least one vowel",0,3);
  if (!opt->cns[0])
    error_msg(stderr,"specify at least one consonant",0,3);
  for (user=opt->first; user; user=user->next) if(!getpwnam(user->name))
    error_msg(stderr,"unknown user specified",0,3);
  if (uid=getuid())
  {
    if (!(pwd=getpwuid(uid)))
      error_msg(stderr,"cannot find current user",0,2);
    if (opt->first)
      error_msg(stderr,"only root can change other user passwords",0,3);
    opt_add_user(opt,pwd->pw_name);
  }
  else if (!opt->first)
    error_msg(stderr,"user(s) must be explicitly specified ",0,3);
}

struct pass_data_t
{
  char  pass[MAX_PASS+1];
  char  new_pass[MAX_PASS+1];
};

char *
ask_pass(void *data, const char *question, int stage)
{
  switch (stage)
    {
     case SETPASS_STAGE_OLD:
         {
            struct termio  term_before, term_tmp;
            int            have_tty;
            char          *ret, *pos, *end;
            fprintf(stderr,"%s",question);
            fflush(stdin);
            if (have_tty=isatty(STDIN_FILENO))
            {
               if (ioctl(STDIN_FILENO,TCGETA,&term_before)) return 0;
               memcpy(&term_tmp, &term_before, sizeof(term_tmp));
               term_tmp.c_iflag&=~(INLCR|ICRNL|IUCLC|ISTRIP|IXON|BRKINT);
               term_tmp.c_lflag&=~(ICANON|ISIG|ECHO);
               term_tmp.c_cc[VTIME]=1;
               term_tmp.c_cc[VMIN]=1;
               if (ioctl(STDIN_FILENO,TCSETAF,&term_tmp)) return 0;
            }
            pos=((struct pass_data_t *)data)->pass;
            end=pos+MAX_PASS;
            while ((pos<end) && (read(STDIN_FILENO,pos,1)==1)
                   && (*pos!='\n') && (*pos!='\r'))
              ++pos;
            *pos=0;
            if (have_tty) (ioctl(STDIN_FILENO,TCSETAF,&term_before));
            fflush(stdin);
            fprintf(stderr,"\n");
            return ((struct pass_data_t *)data)->pass;
         }
     case SETPASS_STAGE_NEW:
     case SETPASS_STAGE_NEW_AGAIN:
       strcpy(((struct pass_data_t *)data)->pass,
              ((struct pass_data_t *)data)->new_pass);
       return ((struct pass_data_t *)data)->pass;
     case SETPASS_STAGE_MESSAGE:
       fprintf(stderr,"%s\n",question);
    }
  return 0;
}

void
init_random (unsigned int   seed)
{
  static char rndstate[RND_SIZE];
  if (seed==0)
  {
    struct timeval tv;
    struct timezone tz;
    tz.tz_minuteswest=0;
    tz.tz_dsttime=0;
    gettimeofday(&tv,&tz);
    init_with_object(&seed,tv);
    init_with_commnd(&seed,ENTHROPY_COMMAND);
  }
  initstate(seed,rndstate,RND_SIZE);
}

int
gen_random (int max)
{
  return (int) (max*(double)(random())/(RAND_MAX+1.0));
}

void
gen_pass (char *pass, size_t size, size_t up, const char *vow, const char *cns)
{
  size_t i, len_v, len_c;
  char   c;
  len_v=strlen(vow);
  len_c=strlen(cns);
  i=size;
  init_random(0);
  while (i>0)
  {
    c=((i%2) ? cns[gen_random(len_c)] : vow[gen_random(len_v)]);
    if (up && ((up>=i) || (gen_random(i)<up)))
    {
      c=toupper(c);
      --up;
    }
    pass[--i]=c;
    if (i && !(i%4)) init_random(0);
  }
  /* final cleanup: terminate string with null and clear local valiables
   * c and up and also random generator internal status
   */
  pass[size]=c=0;
  up=0;
  init_random(1);
}

int
main (int argc, char **argv)
{
  struct pass_data_t  pass_data;
  struct optval       opt;
  struct user        *user;
  opt_parse(&opt,argc,argv);
  opt_check(&opt);
  for (user=opt.first; user; user=user->next)
  {
    gen_pass (pass_data.new_pass, opt.len, opt.upper, opt.vow, opt.cns);
    if (!opt.show
        && (set_pass(&pass_data,user->name,&ask_pass,0)!=PAM_SUCCESS))
      error_msg(stderr,"cannot set password",0,3);
    else if (opt.quiet) printf ("%s\n",pass_data.new_pass);
    else printf ("Username: %s\nPassword: %s\n",user->name,pass_data.new_pass);
    bzero(&pass_data,sizeof(pass_data));
  }
  return 0;
}

