/*
 * textrope (C) 2002 Bartosz Lis <bartoszl@ics.p.lodz.pl>
 *
 * This is a utility library, that helps gather multiple substrings
 * and produce a single string
 */

#define _GNU_SOURCE

#include <stdlib.h>
#include <string.h>

#include "textrope.h"

void
textrope_destroy(textrope_t **tr)
{
  textrope_t *tdel;
  if (tr) while (*tr)
  {
    tdel=*tr;
    *tr=(*tr)->next;
    free(tdel);
  }
}

void
textrope_purge(textrope_t **tr)
{
  textrope_t *tdel;
  if (tr) while (*tr)
  {
    tdel=*tr;
    *tr=(*tr)->next;
    bzero(tdel,sizeof(textrope_t)+tdel->len);
    free(tdel);
  }
}

int
textrope_append(textrope_t **tr, size_t len, const char *str)
{
  textrope_t *tnew;
  if (!tr) return 0;
  if (!len) if (!str || !(len=strlen(str))) return 1;
  if (!(tnew=(textrope_t *)malloc(sizeof(textrope_t)+len))) return 0;
  tnew->next=0;
  tnew->len=len;
  strncpy(tnew->str,str,len);
  tnew->str[len]=0;
  while (*tr) tr=&((*tr)->next);
  *tr=tnew;
  return 1;
}

int
textrope_join(textrope_t **tr1, textrope_t **tr2)
{
  if (!tr1) return 0;
  if (!tr2 || !*tr2) return 1;
  while (*tr1) tr1=&((*tr1)->next);
  *tr1=*tr2;
  *tr2=0;
  return 1;
}

int
textrope_esc(textrope_t **tr, const char *str, char esc, char *spec)
{
  textrope_t *tnew=0;
  const char *p, *q;
  char        buff[2];
  int         ok=1;
  buff[0]=esc;
  q=str;
  while (ok && q && *(p=q)) if (!(q=strpbrk(p,spec)))
    ok=textrope_append(&tnew,0,p);
  else if ((p==q) || (ok=textrope_append(&tnew,q-p,p)))
  {
    buff[1]=*q++;
    ok=textrope_append(&tnew,2,buff);
  }
  if (ok) textrope_join(tr,&tnew); else textrope_purge(&tnew);
  return ok;
}

size_t
textrope_len(const textrope_t *tr)
{
  size_t len=0;
  while (tr)
  {
    len+=tr->len;
    tr=tr->next;
  }
  return len;
}

size_t
textrope_substr(const textrope_t *tr, size_t pos, size_t len, char *dst)
{
  size_t p=0, l;
  while (tr && (tr->len<=pos))
  {
    pos-=tr->len;
    tr=tr->next;
  }
  while (tr && len)
  {
    l=tr->len-pos;
    if (l>len) l=len;
    if (l) strncpy(dst+p,tr->str+pos,l);
    len-=l;
    pos=0;
    p+=l;
    tr=tr->next;
  }
  dst[p]=0;
  return p;
}

char *
textrope_dup(const textrope_t *tr)
{
  char   *str;
  size_t  len;
  if (str=(char *)malloc((len=textrope_len(tr))+1))
    textrope_substr(tr,0,len,str);
  return str;
}

int
textrope_cmp(const textrope_t *tr, const char *str)
{
  int ret;
  if (str) while (tr && *str)
  {
    if (tr->len && (ret=strncmp(tr->str,str,tr->len))) return ret;
    str+=tr->len;
    tr=tr->next;
  }
  return str && *str ? -1 : tr && *tr->str;
}

int
textrope_ncmp(const textrope_t *tr, size_t pos, size_t len, const char *str)
{
  size_t l;
  int    ret;
  if (str)
  {
    while (tr && (tr->len<=pos))
    {
      pos-=tr->len;
      tr=tr->next;
    }
    while (tr && len && *str)
    {
      l=tr->len-pos;
      if (l>len) l=len;
      if (l && (ret=strncmp(tr->str+pos,str,l))) return ret;
      len-=l;
      pos=0;
      str+=l;
      tr=tr->next;
    }
  }
  if (!tr || !*tr->str) len=0;
  return str && *str ? -1 : len>0;
}
