/*
 * Copyright (C) 2002 Bartosz Lis <bartoszl@ics.p.lodz.pl>
 * This is the main module
 */

#define _GNU_SOURCE

#include "vars.h"
#include "textrope.h"

#include <string.h>
#include <stdlib.h>
#include <ctype.h>

const char *
get_var(variable_t *vars, const char *name)
{
  variable_t *vbase;
  if (!name || !*name) return 0;
  if (vars) for (vbase=vars; vbase; vbase=vbase->next)
    if (!strcmp(vbase->name,name)) return vbase->value;
  return 0;
}

const char *
get_var_len(variable_t *vars, const char *name, size_t len)
{
  variable_t *vbase;
  if (!name || !*name) return 0;
  if (vars) for (vbase=vars; vbase; vbase=vbase->next)
    if (!strncmp(name,vbase->name,len)) return vbase->value;
  return 0;
}

int
set_var(variable_t **vars, const char *name, const char *value)
{
  variable_t *vset=0, **vbase;
  char       *val;
  if (!vars || !name || !*name) return 1;
  for (vbase=vars; *vbase; vbase=&((*vbase)->next))
    if (!strcmp((*vbase)->name,name)) break;
  if (*vbase) vset=*vbase;
  else
  {
    if (!(vset=(variable_t *)malloc(sizeof(variable_t)))) return -1;
    if (!(vset->name=strdup(name)))
    {
      free(vset);
      return -1;
    }
    vset->next=0;
  }
  if (!val || !*val) val=0;
  else if (!(val=strdup(value)))
  {
    if (!*vbase)
    {
      free(vset->name);
      free(vset);
    }
    return -1;
  }
  if (*vbase) free((*vbase)->value); else *vbase=vset;
  vset->value=val;
  return 0;
}

int
drop_var(variable_t **vars, const char *name)
{
  variable_t *vdel, **vbase;
  if (!name || !*name) return 1;
  if (vars) for (vbase=vars; *vbase; vbase=&((*vbase)->next))
    if (!strcmp((*vbase)->name,name))
    {
      vdel=*vbase;
      *vbase=(*vbase)->next;
      free(vdel->value);
      free(vdel->name);
      free(vdel);
      return 0;
    }
  return 1;
}

int
purge_var(variable_t **vars, const char *name)
{
  variable_t *vdel, **vbase;
  if (!name || !*name) return 1;
  if (vars) for (vbase=vars; *vbase; vbase=&((*vbase)->next))
    if (!strcmp(((*vbase)->name),name))
    {
      vdel=*vbase;
      *vbase=(*vbase)->next;
      bzero(vdel->name,strlen(vdel->name));
      bzero(vdel->value,strlen(vdel->value));
      free(vdel->value);
      free(vdel->name);
      free(vdel);
      return 0;
    }
  return 1;
}

void
drop_vars(variable_t **vars)
{
  variable_t *vdel;
  if (vars) while (*vars)
  {
    vdel=*vars;
    *vars=(*vars)->next;
    free(vdel->value);
    free(vdel->name);
    free(vdel);
  }
}

void
purge_vars(variable_t **vars)
{
  variable_t *vdel;
  if (vars) while (*vars)
  {
    vdel=*vars;
    *vars=(*vars)->next;
    bzero(vdel->name,strlen(vdel->name));
    bzero(vdel->value,strlen(vdel->value));
    free(vdel->value);
    free(vdel->name);
    free(vdel);
  }
}
#include <stdio.h>
static char *
textrope_substitute(textrope_t **tr, const char *str, variable_t *vars)
{
  const char *p, *q;
  char        c, delim;
  int         ok=1;
  q=str;
  while (ok && q && *(p=q)) if (!(q=strchr(p,':')))
    ok=textrope_append(tr,0,p);
  else if ((p==q) || (ok=textrope_append(tr,q-p,p)))
  {
    if ((delim=*++q)=='{') ++q; else delim=0;
    c=*(p=q);
    if (isalpha(c) || (c=='_')) do c=*++q; while (isalnum(c) || (c=='_'));
    if (delim && ((c!='}') || (p==q)))
    {
      ok=0;
      break;
    }
    if (p==q)
    {
      if (c==':') ++q;
      ok=textrope_append(tr,1,&c);
    }
    else ok=textrope_append(tr,0,get_var_len(vars,p,q-p));
    if (delim) ++q;
  }
  c=0;
  return ok ? textrope_dup(*tr) : 0;
}

char *
substitute(const char *str, variable_t *vars)
{
  textrope_t *tr=0;
  char       *ret;
  if (!str) return 0;
  ret=textrope_substitute(&tr,str,vars);
  textrope_destroy(&tr);
  return ret;
}

char *
safe_substitute(const char *str, variable_t *vars)
{
  textrope_t *tr=0;
  char       *ret;
  if (!str) return 0;
  ret=textrope_substitute(&tr,str,vars);
  textrope_purge(&tr);
  return ret;
}
