/***************************************************************************
                          ann_log.h  -  description
                             -------------------
    begin                : pon kwi 14 2003
    copyright            : (C) 2003 by Bartosz Lis
    email                : bartoszl@ics.p.lodz.pl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef __ANN_LOG_H
#define __ANN_LOG_H

#include <ctype.h>

#include <ann_term.h>
#include <ann_time.h>

namespace ANN
{

  class Log;
  class Log_rec;
  class LS;

  //           //
 // class Log //
//           //

  class Log
  {
  protected:
    LS *children;

    virtual void new_line(int ch_marg, const char *label)=0;
    virtual void log_chr(char chr)=0;
    virtual void log_str(const char *str)=0;
    virtual void log_int(int i)=0;
    virtual void log_uint(unsigned int u)=0;
    virtual void log_double(double d)=0;
    virtual void log_loc(const Loc &l)=0;
    virtual void log_size(const Size &l)=0;
    virtual void log_term(const Term &t, const char *label, size_t which,
                          const Loc *where);

  public:
    Log() : children(0) {}
    virtual ~Log();

    virtual Time   &get_time() const=0;
    virtual char    get_separator() const=0;
    virtual char    get_sublevel_beg() const=0;
    virtual char    get_sublevel_end() const=0;
    virtual size_t  get_margin() const=0;
    virtual bool    attach(Log_rec &child);
    virtual bool    detach(Log_rec &child);
    virtual bool    is_active() const;
    /* tells whether logging is possible */
    virtual void    commence(const char *label_=0);
    /* restarts logging */
    virtual void    open(size_t epoch);
    /* caled whenever epoch starts */
    virtual void    feed(size_t which_, const Loc *where_);
    /* caled whenever step starts */
    virtual void    close(size_t epoch);
    /* caled whenever epoch ends */
    virtual void    conclude(size_t epoch);
    /* caled whenever teaching finishes */
    virtual void    concluded(size_t epoch);
    /* caled whenever teaching is finished */
    virtual Log    &flush()=0;
    /* synchronizes buffers with storage */

    Log &nl(int ch_marg=0, const char *label=0)
    { new_line(ch_marg, label); return *this; }
    Log &nl(const char *label) { new_line(0, label); return *this; }
    Log &log(char chr) { log_chr(chr); return *this; }
    Log &log(const char *str) { log_str(str); return *this; }
    Log &log(int i) { log_int(i); return *this; }
    Log &log(unsigned int u) { log_uint(u); return *this; }
    Log &log(double d) { log_double(d); return *this; }
    Log &log(const Loc &loc) { log_loc(loc); return *this; }
    Log &log(const Size &size) { log_size(size); return *this; }
    Log &log(const Term &t, const char *label=0, size_t which=size_max,
             const Loc *where=0)
    { log_term(t,label,which,where); return *this; }

    Log &operator << (char chr) { return log(chr); }
    Log &operator << (const char *str) { return log(str); }
    Log &operator << (int i) { return log(i); }
    Log &operator << (unsigned int u) { return log(u); }
    Log &operator << (double d) { return log(d); }
    Log &operator << (const Loc &loc) { return log(loc); }
    Log &operator << (const Size &size) { return log(size); }
    Log &operator << (const Term &t) { return log(t); }
  };

  //               //
 // class Log_rec //
//               //

  class Log_rec : public Log
  {
  private:
    Log_rec();
    Log_rec(Log_rec &that);

  protected:
    Log    *orig;
    bool    active;

    virtual void new_line(int ch_marg, const char *label);
    virtual void log_chr(char chr);
    virtual void log_str(const char *str);
    virtual void log_int(int i);
    virtual void log_uint(unsigned int u);
    virtual void log_double(double d);
    virtual void log_loc(const Loc &l);
    virtual void log_size(const Size &l);
    virtual void log_term(const Term &t, const char *label, size_t which,
                          const Loc *where);
    virtual void prop_commence();
    virtual void prop_open(size_t epoch);
    virtual void prop_feed(size_t which_, const Loc *where_);
    virtual void prop_close(size_t epoch);
    virtual void prop_conclude(size_t epoch);
    virtual void prop_concluded(size_t epoch);

  public:
    Log_rec(Log &orig_) : orig(&orig_), active(false) { orig->attach(*this); }
    virtual ~Log_rec();

    virtual Time   &get_time() const;
    virtual char    get_separator() const;
    virtual char    get_sublevel_beg() const;
    virtual char    get_sublevel_end() const;
    virtual size_t  get_margin() const;
    virtual bool    is_active() const;
    virtual void    commence(const char *label_=0);
    virtual void    open(size_t epoch);
    virtual void    feed(size_t which_, const Loc *where_);
    virtual void    close(size_t epoch);
    virtual void    conclude(size_t epoch);
    virtual void    concluded(size_t epoch);
    virtual Log    &flush();

    friend class LS;
  };

  //          //
 // class LS //
//          //

  class LS
  {
  protected:
    LS() {}

    void prop_commence(Log_rec &log) { log.prop_commence(); }
    void prop_open(Log_rec &log, size_t epoch) { log.prop_open(epoch); }
    void prop_feed(Log_rec &log, size_t which_, const Loc *where_)
    { log.prop_feed(which_,where_); }
    void prop_close(Log_rec &log, size_t epoch) { log.prop_close(epoch); }
    void prop_conclude(Log_rec &log, size_t epoch)
    { log.prop_conclude(epoch); }
    void prop_concluded(Log_rec &log, size_t epoch)
    { log.prop_concluded(epoch); }

  public:
    virtual ~LS();
    static LS *create();

    virtual bool  attach(Log_rec &child)=0;
    virtual bool  detach(Log_rec &child)=0;
    virtual void  commence()=0;
    virtual void  open(size_t epoch)=0;
    virtual void  feed(size_t which_, const Loc *where_)=0;
    virtual void  close(size_t epoch)=0;
    virtual void  conclude(size_t epoch)=0;
    virtual void  concluded(size_t epoch)=0;
  };

} // namespace ANN

#endif /* __ANN_LOG_H */
