/***************************************************************************
                          ann_ds.h  -  description
                             -------------------
    begin                : pon kwi 14 2003
    copyright            : (C) 2003 by Bartosz Lis
    email                : bartoszl@ics.p.lodz.pl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef __ANN_DS_H
#define __ANN_DS_H

#include <ctype.h>

#include <ann_loc.h>
#include <ann_term.h>
#include <ann_ts.h>

namespace ANN
{

  class DS;      /* data source */
  class DS_des;  /* data source for desired output */

  //          //
 // class DS //
//          //

  class DS
  {
  private:
    DS(const DS &that);

  protected:
    DS            *des_out;

    DS() : des_out(0) {}
    DS(DS *des_out_) : des_out(des_out_) {}

    virtual bool  do_resize(const Size &sizes_);
    bool          resize(const Size &sizes_){ return do_resize(sizes_); } 
    bool          resize(const Loc &sizes_) { return do_resize(Size(sizes_)); }
    bool          resize(size_t dimension_, size_t *positions_)
    { return do_resize(Size(dimension_,positions_)); }
    bool          resize(size_t total_) { return do_resize(Size(total_)); }
    bool          resize(size_t *positions_=0)
    { return do_resize(Size(positions_)); }
    virtual bool  do_set_length(const Size &len_);
    bool          set_length(const Size &sizes_);
    bool          set_length(const Loc &sizes_)
    { return set_length(Size(sizes_)); }
    bool          set_length(size_t dimension_, size_t *positions_)
    { return set_length(Size(dimension_, positions_)); }
    bool          set_length(size_t total_)
    { return set_length(Size(total_)); }
    bool          set_length(size_t *positions_=0)
    { return set_length(Size(positions_)); }
    virtual bool  do_reset()=0;
    virtual void  do_next()=0;
    virtual bool  do_seek(size_t term_pos_);
    virtual bool  do_locate(const Loc &loc);

    virtual TS   *all_terms();
    TS           *des_terms() { return des_out->all_terms(); }
    TS           *inp_terms() { return input()->all_terms(); }
    bool          des_resize(const Size &sizes_)
    { return des_out->resize(sizes_); }
    void          des_drop()
    { DS *in=input(); if (in->des_out) { delete in->des_out; in->des_out=0; }}

  public:
    virtual ~DS();

    virtual const Size &get_lengths() const=0;
    virtual size_t      get_length() const; /* length of the test/training
                                             * sequence, or size_max if unfinited
                                             * data stream */
    virtual const Size &get_sizes() const; /* term sizes */
    virtual size_t      get_size() const;  /* returns overall term size */
    virtual Term       &term()=0;
    virtual const Term &term() const=0;
    bool                reset();
    void                next();
    bool                seek(size_t term_pos_);
    bool                locate(const Loc &loc);
    /* the above functions call do_xxx() of objects *input() and *des_out
     * forcing both the objects to be synchronized */
    /* the real job of shifting terms is done by functions do_xxx(), that is:
     * reset()/do_reset() -- if possible shifts data source back to the first
     *                       term and returns true otherwise return false
     * next()/do_next() -- shifts data source to the next term, if the last 
     *                     term was presented first is presented again
     * seek()/do_seek() -- shifts data source to the specified term in the
     *                     permutation of terms, term positions in the
     *                     permutation are constant between calls to reset()
     * locate()/do_locate() -- positions at the specified absolute location */
    virtual const Loc  &where() const=0;
    virtual size_t      which() const;
    /* position of the current term in the sequence, the number returned by
     * which() is the position of the term in the permutation of the data
     * source (if terms are permuted), while location returned by where()
     * is always absolute */
    virtual const TS   *all_terms() const;
    /* the above functions return pointer to an array of all terms making up the
     * current training sequence, if such an array can not be provided by the
     * data source, zero is returned (default behavior), in this case next()
     * reset() and seek() functions should be used */
    virtual DS         *input();
    virtual const DS   *input() const;
    /* returns data source containing inputs which is synchronized to this data
     * source (except class DS_des and its descendants, pointer this is
     * returned) */
    DS                 *des_output() { return des_out; }
    const DS           *des_output() const { return des_out; }
    /* returns data source containing desired outputs which is synchronized to
     * this data source or 0 if nothing available */
    /* the value returned by the four above functinos should point to the same
     * object if we call input source object's input() or desired output
     * object's des_output(), otherwise pointer to the other synchronised
     * obcject is returned */

    double &operator [] (size_t idx) { return term()[idx]; }
    double  operator [] (size_t idx) const { return term()[idx]; }
    double &operator [] (const Loc &loc) { return term()[loc]; }
    double  operator [] (const Loc &loc) const { return term()[loc]; }
    DS     &operator ++ () { next(); return *this; }
  };

  //              //
 // class DS_des //
//              //

  class DS_des : public virtual DS
  {
  private:
    DS_des();

  protected:
    DS   *in;

    DS_des(DS &in_) : DS(this), in(&in_) {}

  public:
    virtual ~DS_des();

    virtual const Size &get_lengths() const;
    virtual size_t      get_length() const; 
    virtual const Loc  &where() const;
    virtual size_t      which() const;
    virtual DS         *input();
    virtual const DS   *input() const;
  };

} // namespace ANN

#endif /* __ANN_DS_H */
