#ifndef __TEXTROPE_H
#define __TEXTROPE_H

/*
 * textrope (C) 2002 Bartosz Lis <bartoszl@ics.p.lodz.pl>
 *
 * This is a utility library, that helps gather multiple substrings
 * and produce a single string
 */

#include <ctype.h>

/* recurent structure to hold rope's strings */
typedef struct textrope {
  struct textrope *next;   /* rest of the list */
  size_t           len;    /* length of the current element */
  char             str[1]; /* elementary string, this field is only declared
                            * here to be char[1], but as the structure is
                            * allocated with
                            * malloc(sizeof(struct textrope)+len) this field
                            * has actualy len+1 bytes, where len is the
                            * elementary string length;
                            */
} textrope_t;

/* destroys rope of strings
 */
void
textrope_destroy(textrope_t **tr);

/* destroys rope of strings calling bzero to erase any secret
 */
void
textrope_purge(textrope_t **tr);

/* concatenates a string to the end of a rope
 * returned value is true if malloc was successfull
 */
int
textrope_append(textrope_t **tr, size_t len, const char *str);

/* concatenates the second rope to the end of the first rope
 * the second rope should not be then deallocated by the
 */
int
textrope_join(textrope_t **tr1, textrope_t **tr2);

/* concatenates a string to the end of a rope, evry character named in
 * spec is prepended with esc to form escape sequence
 * returned value is true if malloc was successfull
 */
int
textrope_esc(textrope_t **tr, const char *str, char esc, char *spec);

/* counts rope's length
 */
size_t
textrope_len(const textrope_t *tr);

/* copies substring from a rope to a buffer (buffer must be allocated
 * by the programmer)
 */
size_t
textrope_substr(const textrope_t *tr, size_t pos, size_t len, char *dst);

/* allocates buffer enough to produce single string from the rope entire
 * and then copies rope's strings to that buffer
 * the buffer must be freed by the programmer with free()
 */
char *
textrope_dup(const textrope_t *tr);

/* compare content of textrope with a string, returned value have
 * the same meaning as strcmp's
 */

int
textrope_cmp(const textrope_t *tr, const char *str);
int
textrope_ncmp(const textrope_t *tr, size_t pos, size_t len, const char *str);

#endif /*__TEXTROPE_H*/

